---
title: Sdxl API Usage Guide
createdAt: Thu Jul 18 2024 06:12:37 GMT+0000 (Coordinated Universal Time)
updatedAt: Thu Jul 18 2024 13:40:04 GMT+0000 (Coordinated Universal Time)
---

# Sdxl API Usage Guide

## Introduction

This document will guide developers on how to use the aonweb library to call the Sdxl API, A text-to-image generative AI model that creates beautiful images.

## Prerequisites

- Node.js environment
- `aonweb` library installed
- Valid Aonet APPID

## Basic Usage

### 1. Import Required Modules

```js
import { AI, AIOptions } from 'aonweb';
```

### 2. Initialize AI Instance

```js
const ai_options = new AIOptions({
    appId: 'your_app_id_here',
    dev_mode: true
});

const aonweb = new AI(ai_options);
```

### 3. Prepare Input Data Example

```js
const data = {
   input:{
    "width": 768,
    "height": 768,
    "prompt": "An astronaut riding a rainbow unicorn, cinematic, dramatic",
    "refine": "expert_ensemble_refiner",
    "scheduler": "K_EULER",
    "lora_scale": 0.6,
    "num_outputs": 1,
    "guidance_scale": 7.5,
    "apply_watermark": false,
    "high_noise_frac": 0.8,
    "negative_prompt": "",
    "prompt_strength": 0.8,
    "num_inference_steps": 25
  }
};
```

```js
const data = {
   input:{
  "width": 1024,
  "height": 1024,
  "prompt": "A studio photo of a rainbow coloured cat",
  "refine": "expert_ensemble_refiner",
  "scheduler": "KarrasDPM",
  "num_outputs": 1,
  "guidance_scale": 7.5,
  "high_noise_frac": 0.8,
  "prompt_strength": 0.8,
  "num_inference_steps": 50
}
};
```


```js
const data = {
   input:{
      "mask": "https://replicate.delivery/pbxt/JF3Ld3yPLVA3JIELHx1uaAV5CQOyr4AoiOfo6mJZn2fofGaT/dog-mask.png",
      "image": "https://replicate.delivery/pbxt/JF3LddQgRiMM9Q4Smyfw7q7BR9Gn0PwkSWvJjKDPxyvr8Ru0/cool-dog.png",
      "width": 1024,
      "height": 1024,
      "prompt": "An orange cat sitting on a bench",
      "refine": "base_image_refiner",
      "scheduler": "KarrasDPM",
      "num_outputs": 1,
      "guidance_scale": 7.5,
      "high_noise_frac": 0.8,
      "prompt_strength": 0.8,
      "num_inference_steps": 25
    }
};
```




### 4. Call the AI Model

```js
const price = 8; // Cost of the AI call
try {
    const response = await aonweb.prediction("/predictions/ai/sdxl@asiryan", data, price);
    // Handle response
    console.log("Sdxl result:", response);
} catch (error) {
    // Error handling
    console.error("Error generating :", error);
}
```

### Parameter Description

- `width` Number,Width of output image
- `height` Number,Height of output image
- `image` String,Input image for img2img or inpaint mode
- `prompt` String,Provide the prompt required for model inference
- `refine` String,Which refine style to use,Enum:["no_refiner", "expert_ensemble_refiner", "base_image_refiner"]
- `refine_steps` Number,For base_image_refiner, the number of steps to refine, defaults to num_inference_steps
- `scheduler` String,Enum:["DDIM", "DPMSolverMultistep", "HeunDiscrete", "KarrasDPM", "K_EULER_ANCESTRAL", "K_EULER", "PNDM"]
- `lora_scale` Number,LoRA additive scale. Only applicable on trained models.
- `num_outputs` Number,Number of images to output.
- `guidance_scale` Number,Scale for classifier-free guidance
- `apply_watermark` Boolean,Applies a watermark to enable determining if an image is generated in downstream applications. If you have other provisions for generating or deploying images safely, you can use this to disable watermarking.
- `high_noise_frac` Number,For expert_ensemble_refiner, the fraction of noise to use
- `negative_prompt` String,Input Negative Prompt
- `seed` Number,Random seed. Leave blank to randomize the seed
- `prompt_strength` Number,Prompt strength when using img2img / inpaint. 1.0 corresponds to full destruction of information in image
- `num_inference_steps` Number,Number of denoising steps
- `mask` String,Input mask for inpaint mode. Black areas will be preserved, white areas will be inpainted.

### Notes

- Ensure that the provided image URL is publicly accessible and of good quality to achieve the best recognition results.
- The API may take some time to process the input and generate the result, consider implementing appropriate wait or loading states
- Handle possible errors, such as network issues, invalid input, or API limitations.
- Adhere to the terms of use and privacy regulations, especially when handling image samples of others.

### Example Response

The API response will contain the URL of the generated cloned voice or other relevant information. Parse and use the response data according to the actual API documentation.
